using System;
using System.Data;
using System.Reflection;
using gov.va.med.vbecs.ExceptionManagement;
using COMPONENTCLASS = gov.va.med.vbecs.Common.VbecsTables.ComponentClass;
using COMPONENTCLASSPARAMETER = gov.va.med.vbecs.Common.VbecsTables.ComponentClassParameter;
using SPECIMENTESTTHRESHOLD = gov.va.med.vbecs.Common.VbecsTables.SpecimenTestThreshold;

namespace gov.va.med.vbecs.BOL
{

		#region Header

		///<Package>Package: VBECS - VistA Blood Establishment Computer System</Package>
		///<Warning> WARNING: Per VHA Directive $VADIRECTIVE this class should not be modified</Warning>
		///<MedicalDevice> Medical Device #: $MEDDEVICENO</MedicalDevice>
		///<Developers>
		///	<Developer>Luke Meyer</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/21/2003</CreationDate>
		///<Note>The Food and Drug Administration classifies this software as a medical device.  As such, it may not be changed in any way. Modifications to this software may result in an adulterated medical device under 21CFR820, the use of which is considered to be a violation of US Federal Statutes.  Acquiring and implementing this software through the Freedom of information Act requires the implementor to assume total responsibility for the software, and become a registered manufacturer of a medical device, subject to FDA regulations</Note>
		///<summary>BloodComponentClass business object</summary>

		#endregion

	public class BloodComponentClass:BaseBusinessObject
	{
		#region Variables

		///	<summary>
		///	Identifies a specific blood component.
		///	</summary>
		private int _bloodComponentClassId;
		///	<summary>
		///	Name of the blood component class.
		///	</summary>
		private string _bloodComponentName = "";

		private bool _isSpecimenRequired, _isOrderable;
		private int _maximumTransfusionTime;

		private System.Guid _componentClassParameterGuid;

		#endregion	
	
		#region Constructors

		///<Developers>
		///	<Developer>Luke Meyer</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/21/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1427"> 
		///		<ExpectedInput>None</ExpectedInput>
		///		<ExpectedOutput>Valid BloodComponentClass object</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="460"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Empty BloodComponentClass constructor
		/// </summary>
		public BloodComponentClass()
		{
			Clear();
		}

		///<Developers>
		///	<Developer>Krzysztof Dobranowski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>2/26/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="2392"> 
		///		<ExpectedInput>DataRow containing blood component class data</ExpectedInput>
		///		<ExpectedOutput>BloodComponentClass object</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="461"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///	
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Creates BloodComponentClass object and populates it with data from DataRow
		/// </summary>
		/// <param name="dr"></param>
		public BloodComponentClass(DataRow dr)
		{
			this.LoadFromDataRow(dr);
		}
		#endregion
		
		#region Properties

		///<Developers>
		///	<Developer>Carl Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>12/3/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1000"> 
		///		<ExpectedInput>Valid BloodComponentClassID</ExpectedInput>
		///		<ExpectedOutput>Same number of broken rules</ExpectedOutput>
		///	</Case>
		///	<Case type="1" testid ="2260"> 
		///		<ExpectedInput>Invalid BloodComponentClassID</ExpectedInput>
		///		<ExpectedOutput>Same number of broken rules</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		///<summary>
		/// Unique identifier of the blood component class.
		///</summary>
		///TODO:  Change this to be the enum Common.ComponentClass
		public int BloodComponentClassId
		{
			get
			{
				return _bloodComponentClassId;
			}
			set
			{
				// ToDo: Also check values from UC50
				_bloodComponentClassId = value;
				if (value == (int) Common.ComponentClass.RBC || value == (int) Common.ComponentClass.WB)
				{
					this._isSpecimenRequired = true;
				}

				RuleBroken(MethodBase.GetCurrentMethod().Name, (value == System.Int32.MinValue));
			}
		} 

		///<Developers>
		///	<Developer></Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/5/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="7374"> 
		///		<ExpectedInput>Valid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="7375"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// REMOVE ME - WHEN ALL REFERENCES ARE GONE - DONT USE ME
		/// </summary>
		public int BloodComponentClassID
		{
			//TODO: CLEAN ME UP!
			get
			{
				return _bloodComponentClassId;
			}
			set
			{
				// ToDo: Also check values from UC50
				_bloodComponentClassId = value;
				if (value == (int) Common.ComponentClass.RBC || value == (int) Common.ComponentClass.WB)
				{
					this._isSpecimenRequired = true;
				}

				RuleBroken("BloodComponentClassId", (value == System.Int32.MinValue));
			}
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>6/1/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="462"> 
		///		<ExpectedInput>Guid</ExpectedInput>
		///		<ExpectedOutput>Component Class Parameter Guid</ExpectedOutput>
		///	</Case>
		///	<Case type="1" testid ="463"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// ComponentClassParameterGuid
		/// </summary>
		public System.Guid ComponentClassParameterGuid
		{
			get
			{
				return this._componentClassParameterGuid;
			}
			set
			{
				this._componentClassParameterGuid = value;
			}
		}

		///<Developers>
		///	<Developer>Carl Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>12/3/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1001"> 
		///		<ExpectedInput>Valid string</ExpectedInput>
		///		<ExpectedOutput>Valid string</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="464"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		///<summary>
		/// Name of the component class.
		///</summary>
		public string BloodComponentName
		{
			get
			{
				return _bloodComponentName;
			}
			set
			{
				_bloodComponentName = value;
			}
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>6/1/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1055"> 
		///		<ExpectedInput>boolen</ExpectedInput>
		///		<ExpectedOutput>Orderable indicator</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="1087"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		///<summary>
		/// Identifies if the component class is orderable.
		///</summary>
		public bool IsOrderable
		{
			get
			{
				return _isOrderable;
			}
			set
			{
				if (value != _isOrderable)
				{
					_isOrderable = value;
					IsDirty = true;
				}
			}
		}

		///<Developers>
		///	<Developer>Mike Smith</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>12/26/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1056"> 
		///		<ExpectedInput>boolean.</ExpectedInput>
		///		<ExpectedOutput>Specimen required indicator</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="1088"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		///<summary>
		/// Identifies if a specimen is required for the component class.
		///</summary>
		public bool IsSpecimenRequired
		{
			get
			{
				return _isSpecimenRequired;
			}
			set
			{
				if (value != _isSpecimenRequired)
				{
					_isSpecimenRequired = value;
					IsDirty = true;
				}
			}
		}

		///<Developers>
		///	<Developer>Mike Smith</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>12/26/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1057"> 
		///		<ExpectedInput>Blood component class object.</ExpectedInput>
		///		<ExpectedOutput>Maximum transfusion time.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="1089"> 
		///		<ExpectedInput>100</ExpectedInput>
		///		<ExpectedOutput>Maximum transfusion time set to 100 and IsDirty - true</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		///<summary>
		/// The maximum transfusion time for the component.
		///</summary>
		public int MaximumTransfusionTime
		{
			get
			{
				return _maximumTransfusionTime;
			}
			set
			{
				if (value > Convert.ToInt32(Common.MaximumTransfusionTime.Maximum))
				{
					string errorMessage = Common.StrRes.SysErrMsg.Common.CannotbeGreaterError(Common.Utility.ParseStringwithSpaces(
											MethodBase.GetCurrentMethod().Name, true), Convert.ToInt32(Common.MaximumTransfusionTime.Maximum).ToString()).ResString;
					throw new BOL.BusinessObjectException(errorMessage);
				}
				//
				RuleBroken( MethodBase.GetCurrentMethod().Name, 
					( ( value < Convert.ToInt32( Common.MaximumTransfusionTime.Minimum ) ) || ( value > Convert.ToInt32( Common.MaximumTransfusionTime.Maximum ) ) ) );
				//
				if (value != _maximumTransfusionTime)
				{
					_maximumTransfusionTime = value;
					IsDirty = true;
				}
			}
		}
		
		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>6/12/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6011"> 
		///		<ExpectedInput>RBC or WB component</ExpectedInput>
		///		<ExpectedOutput>True</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="6012"> 
		///		<ExpectedInput>CRYO, PLT, FFP or OTHER component</ExpectedInput>
		///		<ExpectedOutput>False</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Indicates if the component contains red blood cells
		/// BR_50.03
		/// </summary>
		public bool IncludesRedBloodCells
		{
			get
			{
				return (this.BloodComponentClassId == (int)Common.ComponentClass.RBC ||
					this.BloodComponentClassId == (int)Common.ComponentClass.WB);
			}
		}

		#endregion

		#region Methods

		///<Developers>
		///	<Developer>Carl Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>12/3/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="999"> 
		///		<ExpectedInput>None</ExpectedInput>
		///		<ExpectedOutput>Data table of component class values</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="465"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Returns a DataTable of component class values from database
		/// </summary>
		/// 		/// <returns></returns>
		public static DataTable GetComponentClassValues() 
		{
			return DAL.BloodComponentClass.GetComponentClassValues();
		}

		///<Developers>
		///	<Developer>Carl Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>6/25/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="2957"> 
		///		<ExpectedInput>None</ExpectedInput>
		///		<ExpectedOutput>Valid data table</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="466"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		///Returns a data table of component requirements 
		/// </summary>
		/// <returns></returns>
		public static DataTable GetComponentRequirements() 
		{
			return DAL.BloodComponentClass.GetComponentRequirements();
		}

		///<Developers>
		///	<Developer>Mike Smith</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>12/26/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1058"> 
		///		<ExpectedInput>An empty Data row.</ExpectedInput>
		///		<ExpectedOutput>A data row equivalent to the blood component class object.</ExpectedOutput>
		///	</Case>		
		///<Case type="1" testid ="2880"> 
		///		<ExpectedInput>Invalid DataRow</ExpectedInput>
		///		<ExpectedOutput>ArgumentException</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///	<summary>
		///  Loads a data row from an object
		/// </summary>
		public sealed override System.Data.DataRow LoadDataRowFromThis(System.Data.DataRow dtRow)
		{
			if (dtRow.Table.Columns.Contains(COMPONENTCLASS.ComponentClassId))
				dtRow[COMPONENTCLASS.ComponentClassId] = this.BloodComponentClassId;
			
			if (dtRow.Table.Columns.Contains(COMPONENTCLASS.ComponentClassName))
				dtRow[COMPONENTCLASS.ComponentClassName] = this.BloodComponentName;

			if (dtRow.Table.Columns.Contains(COMPONENTCLASSPARAMETER.OrderableIndicator))
				dtRow[COMPONENTCLASSPARAMETER.OrderableIndicator] = this.IsOrderable;
			
			if (dtRow.Table.Columns.Contains(COMPONENTCLASSPARAMETER.SpecimenRequiredIndicator))
				dtRow[COMPONENTCLASSPARAMETER.SpecimenRequiredIndicator] = this.IsSpecimenRequired;
			
			if (dtRow.Table.Columns.Contains(COMPONENTCLASSPARAMETER.MaximumTransfusionTime))
				dtRow[COMPONENTCLASSPARAMETER.MaximumTransfusionTime] = this.MaximumTransfusionTime;
			
			if (dtRow.Table.Columns.Contains(COMPONENTCLASSPARAMETER.RowVersion))
			{
				if (this.RowVersion != null)
				{
					dtRow[COMPONENTCLASSPARAMETER.RowVersion] = this.RowVersion;
				}
			}
			return dtRow;
		}

		///<Developers>
		///	<Developer>Mike Smith</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>12/26/2002</CreationDate>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Loads an object from a data row
		/// </summary>
		protected sealed override  void LoadFromDataRow(System.Data.DataRow dtRow)
		{
			if (dtRow.Table.Columns.Contains(COMPONENTCLASS.ComponentClassId))
                this.BloodComponentClassId = Convert.ToInt32(dtRow[COMPONENTCLASS.ComponentClassId].ToString());

			if (dtRow.Table.Columns.Contains(COMPONENTCLASS.ComponentClassName))
				this.BloodComponentName = dtRow[COMPONENTCLASS.ComponentClassName].ToString();
			
			if (dtRow.Table.Columns.Contains(COMPONENTCLASSPARAMETER.OrderableIndicator))
			{
				if (dtRow.IsNull(COMPONENTCLASSPARAMETER.OrderableIndicator))
					this.IsOrderable = false;
				else
					this.IsOrderable = (bool)dtRow[COMPONENTCLASSPARAMETER.OrderableIndicator];
			}
			
			// This has to be done because even though RBCs and WBs always require a specimen, this is not loaded by default into the DB
			if (this._bloodComponentClassId == (int) Common.ComponentClass.RBC ||
				this._bloodComponentClassId == (int) Common.ComponentClass.WB)
			{
				this._isSpecimenRequired = true;
			}
			else
			{
				if (dtRow.Table.Columns.Contains(COMPONENTCLASSPARAMETER.SpecimenRequiredIndicator))
				{
					if (dtRow.IsNull(COMPONENTCLASSPARAMETER.SpecimenRequiredIndicator))
						this.IsSpecimenRequired = false;
					else
						this.IsSpecimenRequired = (bool)dtRow[COMPONENTCLASSPARAMETER.SpecimenRequiredIndicator];
				}
			}

			if (dtRow.Table.Columns.Contains(COMPONENTCLASSPARAMETER.ComponentClassParameterGuid))
			{
				if (dtRow.IsNull(COMPONENTCLASSPARAMETER.ComponentClassParameterGuid))
					this._componentClassParameterGuid = System.Guid.NewGuid();
				else
					this._componentClassParameterGuid = (System.Guid)dtRow[COMPONENTCLASSPARAMETER.ComponentClassParameterGuid];
			}

			if (dtRow.Table.Columns.Contains(COMPONENTCLASSPARAMETER.MaximumTransfusionTime))
			{
				if (dtRow.IsNull(COMPONENTCLASSPARAMETER.MaximumTransfusionTime))
					this.MaximumTransfusionTime = Convert.ToInt32(Common.MaximumTransfusionTime.Default);
				else
					this.MaximumTransfusionTime = Convert.ToInt32(dtRow[COMPONENTCLASSPARAMETER.MaximumTransfusionTime].ToString());
			}

			if (dtRow.Table.Columns.Contains(COMPONENTCLASS.RowVersion))
			{
				if (dtRow.IsNull(COMPONENTCLASS.RowVersion))
					this.RowVersion = null;
				else
					this.RowVersion = (byte[])dtRow[COMPONENTCLASS.RowVersion];
			}

			this.IsDirty = false;
			this.IsNew = false;
		}



		///<Developers>
		///	<Developer>Mike Smith</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>12/17/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1048"> 
		///		<ExpectedInput>Blood component class object.</ExpectedInput>
		///		<ExpectedOutput>Blood component class object with all it's properties reset.</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="467"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Resets object to initial values
		/// </summary>
		public void Clear()
		{
			ComponentClassParameterGuid = System.Guid.NewGuid();
			BloodComponentClassId = System.Int32.MinValue;
			BloodComponentName = String.Empty;
			IsSpecimenRequired = false;
			MaximumTransfusionTime = System.Int32.MinValue;
			IsOrderable = false;
			IsDirty = false;
			IsNew = true;
		}


		///<Developers>
		///	<Developer>Mike Smith</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>12/19/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1050"> 
		///		<ExpectedInput>None</ExpectedInput>
		///		<ExpectedOutput>Data table of component classes for a division</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="468"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		///<summary>
		/// Calls data access layer to component classes within a division.
		/// Implements BR_50.01
		///</summary>
		public DataTable GetComponentClasses()
		{
			return DAL.BloodComponentClass.GetComponentClasses();
		}

		///<Developers>
		///	<Developer>Mike Smith</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>12/20/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1054"> 
		///		<ExpectedInput>Valid data tables of component classes and lab tests</ExpectedInput>
		///		<ExpectedOutput>None</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3326"> 
		///		<ExpectedInput>Data table with invalid row version</ExpectedInput>
		///		<ExpectedOutput>Row Version Exception thrown</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		///  Calls the data access layer to save the changes in the _ComponentClasses data table.
		/// </summary>
		/// <param name="componentClasses">_ComponentClasses</param>
		/// <param name="labTests">labTests</param>
		/// <param name="lastUpdateFunctionId">lastUpdateFunctionId</param>
		public void SaveComponentClasses(DataTable componentClasses, DataTable labTests, Common.UpdateFunction lastUpdateFunctionId)
		{
			System.Collections.ArrayList _dtArrayList = new System.Collections.ArrayList();
			System.Collections.ArrayList _sprocArrayList = new System.Collections.ArrayList();
			//
			if (componentClasses != null)
			{
				DataTable modified = componentClasses.GetChanges(DataRowState.Modified);
				if (modified!= null)
				{
					modified = Common.Utility.AppendLastUpdateInformation(modified, lastUpdateFunctionId);
					//
					if (modified.Columns.Contains(COMPONENTCLASS.ComponentClassName))
					{
						modified.Columns.Remove(COMPONENTCLASS.ComponentClassName);
					}
					if (modified.Columns.Contains(COMPONENTCLASS.ComponentClassShortName))
					{
						modified.Columns.Remove(COMPONENTCLASS.ComponentClassShortName);
					}
					if (modified.Columns.Contains(COMPONENTCLASS.SortColumn))
					{
						modified.Columns.Remove(COMPONENTCLASS.SortColumn);
					}
					//
					_dtArrayList.Add(modified);
					_sprocArrayList.Add(Common.VbecsStoredProcs.UpdateComponentClassParameter.StoredProcName);
				}
				DataTable added = componentClasses.GetChanges(DataRowState.Added);
				if (added!=null)
				{
					added = Common.Utility.AppendLastUpdateInformation(added, lastUpdateFunctionId);
					//
					if (added.Columns.Contains(COMPONENTCLASS.ComponentClassName))
					{
						added.Columns.Remove(COMPONENTCLASS.ComponentClassName);
					}
					if (added.Columns.Contains(COMPONENTCLASS.ComponentClassShortName))
					{
						added.Columns.Remove(COMPONENTCLASS.ComponentClassShortName);
					}
					if (added.Columns.Contains(COMPONENTCLASS.SortColumn))
					{
						added.Columns.Remove(COMPONENTCLASS.SortColumn);
					}
					//
					_dtArrayList.Add(added);
					_sprocArrayList.Add(Common.VbecsStoredProcs.InsertComponentClassParameter.StoredProcName);
				}
			}

			if (labTests != null)
			{
				DataTable modified = labTests.GetChanges(DataRowState.Modified);
				if (modified != null)
				{
					modified = Common.Utility.AppendLastUpdateInformation(modified, lastUpdateFunctionId);
					//
					if (modified.Columns.Contains(Common.VbecsTables.SpecimenType.SpecimenTypeText))
					{
						modified.Columns.Remove(Common.VbecsTables.SpecimenType.SpecimenTypeText);
					}
					//
					_dtArrayList.Add(modified);
					_sprocArrayList.Add(Common.VbecsStoredProcs.UpdateLabTest.StoredProcName);
				}
				DataTable added = labTests.GetChanges(DataRowState.Added);
				if (added != null)
				{
					added = Common.Utility.AppendLastUpdateInformation(added, lastUpdateFunctionId);
					//
					if (added.Columns.Contains(Common.VbecsTables.SpecimenType.SpecimenTypeText))
					{
						added.Columns.Remove(Common.VbecsTables.SpecimenType.SpecimenTypeText);
					}
					//

					//Logic added here because it was removed from the sproc
					foreach(System.Data.DataRow dr in added.Rows)
					{
						if (added.Columns.Contains(SPECIMENTESTTHRESHOLD.ComponentClassId))
						{
							if (!dr.IsNull(SPECIMENTESTTHRESHOLD.ComponentClassId))
							{
								if ((int)dr[SPECIMENTESTTHRESHOLD.ComponentClassId] < 1)
									dr[SPECIMENTESTTHRESHOLD.ComponentClassId] = DBNull.Value;
							}
						}
					}
					_dtArrayList.Add(added);
					_sprocArrayList.Add(Common.VbecsStoredProcs.InsertSpecimenTestThreshold.StoredProcName);
				}
			}
			
			if (componentClasses != null || labTests != null)
			{
				new Common.StoredProcedure().TransactionalGetValue(_sprocArrayList, _dtArrayList);
			}
		}

		#endregion

	}
}





